
## Say screen ##################################################################
##
## The say screen is used to display dialogue to the player. It takes two
## parameters, who and what, which are the name of the speaking character and
## the text to be displayed, respectively. (The who parameter can be None if no
## name is given.)
##
## This screen must create a text displayable with id "what", as Ren'Py uses
## this to manage text display. It can also create displayables with id "who"
## and id "window" to apply style properties.
##
## https://www.renpy.org/doc/html/screen_special.html#say

default persistent.say_window_alpha = 0.85

screen say(who, what):
    style_prefix "say"

    window:
        background Transform(style.window.background, alpha=persistent.say_window_alpha)
        id "window"

        if who is not None:

            window:
                id "namebox"
                style "namebox"
                text who id "who"

        text what id "what"

    ## If there's a side image, display it in front of the text.
    add SideImage() xalign 0.0 yalign 1.0


## Make the namebox available for styling through the Character object.
init python:
    config.character_id_prefixes.append('namebox')

# Style for the dialogue window
style window:
    xanchor 0.0 xpos 0.0
    yalign 1.0
    xysize (1920, 277)
    padding (425, 40, 92, 20)
    background Image("gui/textbox.png", xalign=0.5, yalign=1.0)

# Style for the dialogue
style say_dialogue:
    adjust_spacing False
    ypos 60

# The style for dialogue said by the narrator
style say_thought:
    is say_dialogue

# Style for the box containing the speaker's name
style namebox:
    xpos 20
    xysize (None, None)
    yoffset -10
    background Frame("gui/namebox.png", 5, 5, 5, 5, tile=False, xalign=0.0)
    padding (5, 5, 5, 5)

# Style for the text with the speaker's name
style say_label:
    color '#fff0f5'
    xalign 0.0
    yalign 0.5
    size gui.name_text_size
    font gui.name_text_font


## Quick Menu screen ###########################################################
##
## The quick menu is displayed in-game to provide easy access to the out-of-game
## menus.

screen quick_menu():

    ## Ensure this appears on top of other screens.
    zorder 100

    if quick_menu:

        if nvl_quick_menu:

            hbox:
                style_prefix "quick"

                textbutton _("Save") action ShowMenu('save'):
                    tooltip "Save progress"

                textbutton _("Load") action ShowMenu('load'):
                    tooltip "Load a save"

                textbutton _("Auto") action Preference("auto-forward", "toggle"):
                    tooltip "Auto forward"

                textbutton _("Skip") action Skip() alternate Skip(fast=True, confirm=True):
                    tooltip "Fast forward"

                textbutton _("Back") action Rollback():
                    tooltip "Previous message"
                    
                textbutton _("History") action ShowMenu('history'):
                    tooltip "Log of previous dialogue"

                textbutton _("Settings") action ShowMenu('preferences'):
                    tooltip "Settings"

                textbutton _("Hide") action HideInterface():
                    tooltip "Hide the UI"

        else:
            hbox:
                style_prefix "quickadv"

                imagebutton idle "gui/button/save.png" action ShowMenu('save') at adv_image_button:
                    tooltip _("Save progress")

                imagebutton idle "gui/button/load.png" action ShowMenu('load') at adv_image_button:
                    tooltip _("Load a save")

                imagebutton idle "gui/button/autoforward.png" action Preference("auto-forward", "toggle") at adv_image_button:
                    tooltip _("Auto forward")

                imagebutton idle "gui/button/skip.png" action Skip() alternate Skip(fast=True, confirm=True) at adv_image_button:
                    tooltip _("Fast forward")

                imagebutton idle "gui/button/previousmessage.png" action Rollback() at adv_image_button:
                    tooltip _("Previous message")
                    
                imagebutton idle "gui/button/log.png" action ShowMenu('history') at adv_image_button:
                    tooltip _("Log of previous dialogue")

                imagebutton idle "gui/button/settings.png" action ShowMenu('preferences') at adv_image_button:
                    tooltip _("Settings")

                imagebutton idle "gui/button/hide.png" action HideInterface() at adv_image_button:
                    tooltip _("Hide the UI")

        $ tooltip = GetTooltip()

        if tooltip:
            nearrect:
                focus "tooltip"
                prefer_top True

                frame:
                    xalign 0.5
                    padding (15,10)
                    text tooltip


## This code ensures that the quick_menu screen is displayed in-game, whenever
## the player has not explicitly hidden the interface.
init python:
    config.overlay_screens.append("quick_menu")

default quick_menu = True
default nvl_quick_menu = True

style quick_hbox:
    xalign 0.5 xoffset 15
    yalign 1.0 yoffset -8
    spacing 4

style quickadv_hbox:
    xanchor 1.0 xpos 0.94
    ycenter 0.77 yoffset -3
    spacing 8

style quickadv_image_button:
    ycenter 0.5
    hover_sound "audio/ui/cutter.ogg"
    activate_sound "<volume 0.75>audio/ui/drum_down.ogg"

transform adv_image_button:
    on hover:
        yzoom 1.15
        matrixcolor BrightnessMatrix(value=0.15)
    on idle:
        yzoom 1.0
        matrixcolor BrightnessMatrix(value=0.0)
    

style quick_button:
    background None
    padding (15, 6, 15, 0)

style quick_button_text:
    size 21
    selected_color '#FF2652'
    idle_color "#b4bec8"

## NVL screen ##################################################################
##
## This screen is used for NVL-mode dialogue and menus.
##
## https://www.renpy.org/doc/html/screen_special.html#nvl


screen nvl(dialogue, items=None):

    window:
        style "nvl_window"

        has vbox
        spacing 15

        use nvl_dialogue(dialogue)

        ## Displays the menu, if given. The menu may be displayed incorrectly if
        ## config.narrator_menu is set to True.
        for i in items:

            textbutton i.caption:
                action i.action
                style "nvl_button"

    add SideImage() xalign 0.0 yalign 1.0


screen nvl_dialogue(dialogue):

    for d in dialogue:

        window:
            id d.window_id

            fixed:
                yfit True

                if d.who is not None:

                    text d.who:
                        id d.who_id

                text d.what:
                    id d.what_id


## This controls the maximum number of NVL-mode entries that can be displayed at
## once.
define config.nvl_list_length = 6

# The style for the NVL "textbox"
style nvl_window:
    is default
    xfill True yfill True
    background "gui/nvl.png"
    padding (0, 15, 0, 30)

# The style for the text of the speaker's name
style nvl_label:
    is say_label
    xpos 645 xanchor 1.0
    ypos 0 yanchor 0.0
    xsize 0
    min_width 0
    xmaximum 0
    textalign 1.0

# The style for dialogue in NVL
style nvl_dialogue:
    is say_dialogue
    xcenter 0.5
    ypos 12
    xsize 885
    min_width 885

# The style for dialogue said by the narrator in NVL
style nvl_thought:
    is nvl_dialogue

style nvl_button:
    xpos 675
    xanchor 0.0

style nvl_entry:
    xcenter 0.5 


## Bubble screen ###############################################################
##
## The bubble screen is used to display dialogue to the player when using speech
## bubbles. The bubble screen takes the same parameters as the say screen, must
## create a displayable with the id of "what", and can create displayables with
## the "namebox", "who", and "window" ids.
##
## https://www.renpy.org/doc/html/bubble.html#bubble-screen

screen bubble(who, what):
    style_prefix "bubble"

    window:
        id "window"

        if who is not None:

            window:
                id "namebox"
                style "bubble_namebox"

                text who:
                    id "who"

        text what:
            id "what"

style bubble_window:
    is empty
    xpadding 30
    top_padding 5
    bottom_padding 5

style bubble_namebox:
    is empty
    xalign 0.5

style bubble_who:
    is default
    xalign 0.5
    textalign 0.5
    color "#000"

style bubble_what:
    is default
    align (0.5, 0.5)
    text_align 0.5
    layout "subtitle"
    color "#000"

define bubble.frame = Frame("gui/bubble.png", 55, 55, 55, 95)
define bubble.thoughtframe = Frame("gui/thoughtbubble.png", 55, 55, 55, 55)

define bubble.properties = {
    "bottom_left" : {
        "window_background" : Transform(bubble.frame, xzoom=1, yzoom=1),
        "window_bottom_padding" : 27,
    },

    "bottom_right" : {
        "window_background" : Transform(bubble.frame, xzoom=-1, yzoom=1),
        "window_bottom_padding" : 27,
    },

    "top_left" : {
        "window_background" : Transform(bubble.frame, xzoom=1, yzoom=-1),
        "window_top_padding" : 27,
    },

    "top_right" : {
        "window_background" : Transform(bubble.frame, xzoom=-1, yzoom=-1),
        "window_top_padding" : 27,
    },

    "thought" : {
        "window_background" : bubble.thoughtframe,
    }
}

define bubble.expand_area = {
    "bottom_left" : (0, 0, 0, 22),
    "bottom_right" : (0, 0, 0, 22),
    "top_left" : (0, 22, 0, 0),
    "top_right" : (0, 22, 0, 0),
    "thought" : (0, 0, 0, 0),
}
